USE [UserStore]
GO

DECLARE @@MESSAGE varchar(2000);

IF NOT EXISTS
(
    SELECT TOP 1
        1
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE [TABLE_NAME] = 'UserTable'
          AND [COLUMN_NAME] = 'UniqueUserId'
)
BEGIN

    ALTER TABLE UserTable ADD [UniqueUserId] [uniqueidentifier] NULL
    EXEC xp_logevent 60000,
                     '| Column [UniqueUserId] was added |',
                     informational;
END

IF NOT EXISTS
(
    SELECT TOP 1
        1
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE [TABLE_NAME] = 'UserTable'
          AND [COLUMN_NAME] = 'CookieKeys'
)
BEGIN

    ALTER TABLE UserTable ADD [CookieKeys] VARCHAR(1000) NULL
    EXEC xp_logevent 60000,
                     '| Column [CookieKeys] was added |',
                     informational;
END


IF EXISTS
(
    SELECT *
    FROM sys.objects
    WHERE type = 'P'
          AND OBJECT_ID = OBJECT_ID('dbo.sp_GetUserProfile')
)
    DECLARE @newname varchar(56)
BEGIN
    SET @newname = 'sp_GetUserProfile_backup_' + convert(varchar(50), getdate(), 112)
    EXEC sp_rename 'sp_GetUserProfile', @newname
    EXEC xp_logevent 60000,
                     '| [sp_GetUserProfile] was renamed |',
                     informational;
END

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

CREATE PROC [dbo].[sp_GetUserProfile] @UserName VARCHAR(60)
AS
BEGIN
    DECLARE @UniqueUserId [uniqueidentifier]

    SET @UniqueUserId =
    (
        SELECT UniqueUserId FROM UserTable WHERE UserName = @UserName
    )
    IF @UniqueUserId IS NULL
    BEGIN
        SET @UniqueUserId = newID()
        UPDATE UserTable
        SET UniqueUserId = @UniqueUserId
        WHERE UserName = @UserName
    END

    SELECT UserName,
           IsNull(FirstName, '') FirstName,
           IsNull(LastName, '') LastName,
           IsNull(Phone1, '') Phone1,
           IsNull(Phone2, '') Phone2,
           IsNull(Phone3, '') Phone3,
           IsNull(Phone4, '') Phone4,
           IsNull(Email1, '') Email1,
           IsNull(Email2, '') Email2,
           IsNull(Email3, '') Email3,
           IsNull(Email4, '') Email4,
           IsNull(AuxID1, '') AuxID1,
           IsNull(AuxID2, '') AuxID2,
           IsNull(AuxID3, '') AuxID3,
           IsNull(AuxID4, '') AuxID4,
           IsNull(AuxID5, '') AuxID5,
           IsNull(AuxID6, '') AuxID6,
           IsNull(AuxID7, '') AuxID7,
           IsNull(AuxID8, '') AuxID8,
           IsNull(AuxID9, '') AuxID9,
           IsNull(AuxID10, '') AuxID10,
           IsNull(pinHash, '') pinHash,
           IsNull(Questions, '') Questions,
           IsNull(Answers, '') Answers,
           IsNull(ChallengeQuestion, '') ChallengeQuestion,
           IsNull(ChallengeAnswer, '') ChallengeAnswer,
           IsNull(CertResetDate, '1/1/1900') CertResetDate,
           IsNull(CertCount, '0') CertCount,
           IsNull(CertSerialNumber, '') CertSerialNumber,
           IsNull(MobileResetDate, '1/1/1900') MobileResetDate,
           IsNull(MobileCount, '0') MobileCount,
           IsNull(ExtSyncPwdDate, '1/1/1900') ExtSyncPwdDate,
           IsNull(HardwareToken, '') HardwareToken,
           IsNull(iOSDevices, '') iOSDevices,
           IsNull(OATHSeed, '') OATHSeed,
           IsNull(OneTimeOATHList, '') OneTimeOATHList,
           isNull(CookieKeys, '') CookieKeys,
           IsNull(GroupList, '') GroupList,
           @UniqueUserId UniqueUserId
    FROM UserTable
    WHERE UserName = @UserName

    SELECT DigitalFP
    FROM UserFP
    WHERE UserName = @UserName

    SELECT PNToken
    FROM UserPN
    WHERE UserName = @UserName

    SELECT AccessHistory
    FROM UserAccessHistory
    WHERE UserName = @UserName

    SELECT OATHToken
    FROM UserOT
    WHERE UserName = @UserName
END
GO


IF EXISTS
(
    SELECT *
    FROM sys.objects
    WHERE type = 'P'
          AND OBJECT_ID = OBJECT_ID('dbo.sp_UpdateUserProfile')
)
    DECLARE @newname varchar(56)
BEGIN
    SET @newname = 'sp_UpdateUserProfile_backup_' + convert(varchar(50), getdate(), 112)
    EXEC sp_rename 'sp_UpdateUserProfile', @newname
    EXEC xp_logevent 60000,
                     '| [sp_UpdateUserProfile] was renamed |',
                     informational;
END

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO

CREATE PROC [dbo].[sp_UpdateUserProfile]
    @UserName VARCHAR(60),
    @FirstName VARCHAR(50) = NULL,
    @LastName VARCHAR(50) = NULL,
    @Phone1 VARCHAR(60) = NULL,
    @Phone2 VARCHAR(40) = NULL,
    @Phone3 VARCHAR(40) = NULL,
    @Phone4 VARCHAR(40) = NULL,
    @Email1 VARCHAR(60) = NULL,
    @Email2 VARCHAR(60) = NULL,
    @Email3 VARCHAR(60) = NULL,
    @Email4 VARCHAR(60) = NULL,
    @AuxID1 VARCHAR(512) = NULL,
    @AuxID2 VARCHAR(512) = NULL,
    @AuxID3 VARCHAR(512) = NULL,
    @AuxID4 VARCHAR(512) = NULL,
    @AuxID5 VARCHAR(512) = NULL,
    @AuxID6 VARCHAR(512) = NULL,
    @AuxID7 VARCHAR(512) = NULL,
    @AuxID8 VARCHAR(512) = NULL,
    @AuxID9 VARCHAR(512) = NULL,
    @AuxID10 VARCHAR(512) = NULL,
    @pinHash VARCHAR(120) = NULL,
    @Questions VARCHAR(1000) = NULL,
    @Answers VARCHAR(1000) = NULL,
    @ChallengeQuestion VARCHAR(1000) = NULL,
    @ChallengeAnswer VARCHAR(1000) = NULL,
    @CertResetDate VARCHAR(50) = NULL,
    @CertSerialNumber VARCHAR(1000) = NULL,
    @CertCount VARCHAR(50) = NULL,
    @MobileResetDate VARCHAR(50) = NULL,
    @MobileCount VARCHAR(50) = NULL,
    @ExtSyncPwdDate VARCHAR(50) = NULL,
    @HardwareToken VARCHAR(1000) = NULL,
    @iOSDevices VARCHAR(1000) = NULL,
    @OATHSeed VARCHAR(MAX) = NULL,
    @OneTimeOATHList VARCHAR(1000) = NULL,
    @CookieKeys VARCHAR(1000) = NULL,
    @DigitalFP dbo.ObjectTable READONLY,
    @PNToken dbo.ObjectTable READONLY,
    @AccessHistory dbo.ObjectTable READONLY,
    @OATHToken dbo.ObjectTable READONLY
AS
DECLARE @UniqueUserId [uniqueidentifier] = NEWID()


IF NOT EXISTS (SELECT UserName FROM UserTable WHERE UserName = @UserName)
BEGIN
    INSERT INTO UserTable
    (
        UserName,
        UniqueUserId
    )
    VALUES
    (@UserName, @UniqueUserId)
END

BEGIN TRY
    BEGIN TRANSACTION

    UPDATE UserTable
    SET FirstName = IsNull(@FirstName, FirstName),
        LastName = IsNull(@LastName, LastName),
        Phone1 = IsNull(@Phone1, Phone1),
        Phone2 = IsNull(@Phone2, Phone2),
        Phone3 = IsNull(@Phone3, Phone3),
        Phone4 = IsNull(@Phone4, Phone4),
        Email1 = IsNull(@Email1, Email1),
        Email2 = IsNull(@Email2, Email2),
        Email3 = IsNull(@Email3, Email3),
        Email4 = IsNull(@Email4, Email4),
        AuxID1 = IsNull(@AuxID1, AuxID1),
        AuxID2 = IsNull(@AuxID2, AuxID2),
        AuxID3 = IsNull(@AuxID3, AuxID3),
        AuxID4 = IsNull(@AuxID4, AuxID4),
        AuxID5 = IsNull(@AuxID5, AuxID5),
        AuxID6 = IsNull(@AuxID6, AuxID6),
        AuxID7 = IsNull(@AuxID7, AuxID7),
        AuxID8 = IsNull(@AuxID8, AuxID8),
        AuxID9 = IsNull(@AuxID9, AuxID9),
        AuxID10 = IsNull(@AuxID10, AuxID10),
        pinHash = IsNull(@pinHash, pinHash),
        Questions = IsNull(@Questions, Questions),
        Answers = IsNull(@Answers, Answers),
        ChallengeQuestion = IsNull(@ChallengeQuestion, ChallengeQuestion),
        ChallengeAnswer = IsNull(@ChallengeAnswer, ChallengeAnswer),
        CertResetDate = IsNull(@CertResetDate, CertResetDate),
        CertCount = IsNull(@CertCount, CertCount),
        CertSerialNumber = IsNull(@CertSerialNumber, CertSerialNumber),
        MobileResetDate = IsNull(@MobileResetDate, MobileResetDate),
        MobileCount = IsNull(@MobileCount, MobileCount),
        ExtSyncPwdDate = IsNull(@ExtSyncPwdDate, ExtSyncPwdDate),
        HardwareToken = IsNull(@HardwareToken, HardwareToken),
        iOSDevices = IsNull(@iOSDevices, iOSDevices),
        OATHSeed = IsNull(@OATHSeed, OATHSeed),
        OneTimeOATHList = IsNull(@OneTimeOATHList, OneTimeOATHList),
        CookieKeys = IsNull(@CookieKeys, CookieKeys)
    WHERE UserName = @UserName

    --- Update Fingerprints ---
    IF EXISTS (SELECT ObjectValue FROM @DigitalFP)
    BEGIN
        DELETE FROM UserFP
        WHERE UserName = @UserName

        INSERT INTO UserFP
        SELECT @UserName,
               ObjectValue
        FROM @DigitalFP
        WHERE ObjectValue <> 'ERASE'
    END
    ---------------------------

    --- Update Push Notification ---
    IF EXISTS (SELECT ObjectValue FROM @PNToken)
    BEGIN
        DELETE FROM UserPN
        WHERE UserName = @UserName

        INSERT INTO UserPN
        SELECT @UserName,
               ObjectValue
        FROM @PNToken
        WHERE ObjectValue <> 'ERASE'
    END
    ---------------------------

    --- Update Access History ---
    IF EXISTS (SELECT ObjectValue FROM @AccessHistory)
    BEGIN
        DELETE FROM UserAccessHistory
        WHERE UserName = @UserName

        INSERT INTO UserAccessHistory
        SELECT @UserName,
               ObjectValue
        FROM @AccessHistory
        WHERE ObjectValue <> 'ERASE'
    END
    ---------------------------

    --- Update OATH Token ---
    IF EXISTS (SELECT ObjectValue FROM @OATHToken)
    BEGIN
        DELETE FROM UserOT
        WHERE UserName = @UserName

        INSERT INTO UserOT
        SELECT @UserName,
               ObjectValue
        FROM @OATHToken
        WHERE ObjectValue <> 'ERASE'
    END
    ---------------------------
    COMMIT
END TRY
BEGIN CATCH
    IF @@TRANCOUNT > 0
        ROLLBACK
END CATCH
GO


IF EXISTS
(
    SELECT *
    FROM sys.objects
    WHERE type = 'P'
          AND OBJECT_ID = OBJECT_ID('dbo.sp_CreateUser')
)
    DECLARE @newname varchar(56)
BEGIN
    SET @newname = 'sp_CreateUser_backup_' + convert(varchar(50), getdate(), 112)
    EXEC sp_rename 'sp_CreateUser', @newname
    EXEC xp_logevent 60000, '| [sp_CreateUser] was renamed |', informational;
END

SET ANSI_NULLS ON
GO

SET QUOTED_IDENTIFIER ON
GO


CREATE PROC [dbo].[sp_CreateUser]
    @UserName VARCHAR(60),
    @Password VARCHAR(128),
    @PasswordSalt VARCHAR(128),
    @PasswordFormat int,
    @Status int OUTPUT
AS
BEGIN
    IF NOT EXISTS (SELECT 1 FROM UserTable WHERE UserName = @UserName)
        INSERT INTO UserTable
        (
            [UserName],
            [Password],
            [PasswordSalt],
            [PasswordFormat],
            [UniqueUserId]
        )
        VALUES
        (@UserName, @Password, @PasswordSalt, @PasswordFormat, NEWID())

    IF @@ROWCOUNT > 0
        SELECT @Status = 0
    ELSE
        SELECT @Status = 1
END
GO

IF NOT EXISTS(SELECT TOP 1 1 FROM sys.indexes indexes INNER JOIN sys.objects 
objects ON indexes.object_id = objects.object_id WHERE indexes.name 
='UserIndex' AND objects.name = 'UserTable')
BEGIN
    CREATE NONCLUSTERED INDEX UserIndex ON UserStore.dbo.UserTable (UserName)
    PRINT 'INDEX CREATED [UserIndex] ON [dbo].[UserTable]'
END
GO

IF NOT EXISTS(SELECT TOP 1 1 FROM sys.indexes indexes INNER JOIN sys.objects 
objects ON indexes.object_id = objects.object_id WHERE indexes.name 
='FPIndex' AND objects.name = 'UserFP')
BEGIN
    CREATE NONCLUSTERED INDEX FPIndex ON UserStore.dbo.UserFP (UserName)
    PRINT 'INDEX CREATED [FPIndex] ON [dbo].[UserFP]'
END
GO

IF NOT EXISTS(SELECT TOP 1 1 FROM sys.indexes indexes INNER JOIN sys.objects 
objects ON indexes.object_id = objects.object_id WHERE indexes.name 
='OTIndex' AND objects.name = 'UserOT')
BEGIN
    CREATE NONCLUSTERED INDEX OTIndex ON UserStore.dbo.UserOT (UserName)
    PRINT 'INDEX CREATED [OTIndex] ON [dbo].[UserOT]'
END
GO

IF NOT EXISTS(SELECT TOP 1 1 FROM sys.indexes indexes INNER JOIN sys.objects 
objects ON indexes.object_id = objects.object_id WHERE indexes.name 
='AccessIndex' AND objects.name = 'UserAccessHistory')
BEGIN
    CREATE NONCLUSTERED INDEX AccessIndex ON UserStore.dbo.UserAccessHistory (UserName)
    PRINT 'INDEX CREATED [AccessIndex] ON [dbo].[UserAccessHistory]'
END
GO

IF NOT EXISTS(SELECT TOP 1 1 FROM sys.indexes indexes INNER JOIN sys.objects 
objects ON indexes.object_id = objects.object_id WHERE indexes.name 
='PNIndex' AND objects.name = 'UserPN')
BEGIN
    CREATE NONCLUSTERED INDEX PNIndex ON UserStore.dbo.UserPN (UserName)
    PRINT 'INDEX CREATED [PNIndex] ON [dbo].[UserPN]'
END
GO

GRANT EXECUTE ON [UserStore].dbo.sp_GetUserProfile TO db_serviceaccount
GRANT EXECUTE ON [UserStore].dbo.sp_UpdateUserProfile TO db_serviceaccount
GRANT EXECUTE ON [UserStore].dbo.sp_CreateUser TO db_serviceaccount